#!/usr/local/bin/python
# -*- coding: utf-8 -*-

# Gorazd Generator
# Generator of dictionary entries from ALTO XML.
# Copyright (C) 2018  Vít Tuček, Slovanský ústav AV ČR, v. v. i.

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

import argparse
import sys

from generator import SJSGenerator, RSIGenerator, SNSPGenerator
from utils import run_bibupload, init_logs, handle_exception

LOG_DIR = 'log'  # default logging directory


def main():
    parser = argparse.ArgumentParser(description='SJSGenerator of vocabulary records from plain text / ALTO XML.')
    parser.add_argument("input_dir", help='Directory containing ALTO XML files for input.')
    parser.add_argument("output_dir", help='Directory containing generated files.')
    parser.add_argument('-g', "--generator", help="Type of generator: SJS, RSI, SNSP", default="SJS")
    parser.add_argument("-p", "--preprocess",
                        help="If preprocessing is turned on we fix slavonic and greek characters and join words"
                             " that were split at line breaks.",
                        action='store_true', default=True)
    parser.add_argument("-l", "--log_dir", help="Directory for log files. Default is the current working directory.",
                        default=LOG_DIR)
    parser.add_argument("-r", "--run_bibupload", help="Run bibupload.", action='store_true', default=False)
    args = parser.parse_args()
    if args.generator == "SJS":
        Generator = SJSGenerator
    elif args.generator == "RSI":
        Generator = RSIGenerator
    elif args.generator == "SNSP":
        Generator = SNSPGenerator
    else:
        print("Wrong generator!")
        sys.exit(1)
    init_logs(args.generator + "Generator", log_dir=args.log_dir)
    gen = Generator(input_dir=args.input_dir, output_dir=args.output_dir, preprocess=args.preprocess)
    gen.log.info("The program has been run with the following arguments: %s" % vars(args))
    sys.excepthook = handle_exception

    gen.run()
    if args.run_bibupload:
        run_bibupload(gen)


if __name__ == "__main__":
    main()
